using System;
using System.Reflection;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages;

namespace gov.va.med.vbecs.DAL.VistALink.Client
{
	/// <summary>
	/// Base client class for connecting to the VistA and executing remote procedure calls. 
	/// Applications must derive their own specific version of this class in order 
	/// use application-specific UI. In order to do this, certain abstract methods must be
	/// implemented with respect to the UI used. Standard implementations for GUI and CLI are
	/// available and may be used in applications. 
	/// 
	/// Developers are highly encouraged to refer to this base class in the code, rather than to 
	/// UI-specific implementation. This will allow for easy change in UI if required.
	/// </summary>
	public abstract class RpcBroker : IDisposable
	{
		private VistALinkClientConnection _vlConnection;
		private bool _isLoggedOn;

		private RemoteServerSetupAndIntroInfo _remoteServerInfo;
		private RemoteUserInfo _remoteUserInfo;
		private DivisionInfo _remoteUserLogonDivisionInfo;
		private RemoteUserNameInfo _remoteUserNameInfo;		

		// Non-resettable members (not reset when broker is disconnected). 
		private bool _reuseAccessAndVerifyCodes;
		private VistASecurityCode _cachedAccessCode;
		private VistASecurityCode _cachedVerifyCode;
		private ServerConnectionInfo _cachedServerConnectionInfo;

        // Logger object
        readonly ILogger _logger = LogManager.Instance().LoggerLocator.GetLogger(MethodBase.GetCurrentMethod().DeclaringType);

		/// <summary>
		/// Connection state change event occurring when VistALink either logged on or off,
		/// or coldly disconnected. 
		/// </summary>
		public event VistALinkConnectionStateChangedDelegate ConnectionStateChanged;

		/// <summary>
		/// Default constructor initializing an object.
		/// </summary>
		protected RpcBroker()
		{
			ResetConnectionStateVars();
			ResetCachedAccessCodes();
		}
	
		/// <summary>
		/// Connects to VistA server specified with <see cref="ServerConnectionInfo"/> parameter.
		/// </summary>
		/// <param name="serverConnectionInfo">VistA server to connect to.</param>
		private void Connect( ServerConnectionInfo serverConnectionInfo )
		{
			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

            _logger.Debug(string.Format("Connect is called. IP: {0}, PORT: {1}", serverConnectionInfo.IPAddress, serverConnectionInfo.PortNumber));

			try
			{
				_vlConnection = new VistALinkClientConnection( serverConnectionInfo );
				_vlConnection.ConnectionStateChanged += ConnectionStateChangedHandler;
				_vlConnection.Connect();
			}
			catch
			{
                _logger.Error("Connection to VistA failed. Close and re-throw exception");
				Close();
				throw;
			}					
		}

		/// <summary>
		/// Connects and logs on to M server specified in <see cref="ServerConnectionInfo"/> object.
		/// </summary>
		/// <param name="serverConnectionInfo">Server connection info.</param>
		/// <returns>Returns false if users cancels logon. True on successful logon.</returns>
		public bool Logon( ServerConnectionInfo serverConnectionInfo )
		{
			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

			return ConnectAndLogon( serverConnectionInfo );
		}

		/// <summary>
		/// Gets M server connection information from global VistALink config, 
		/// connects and logs on. 
		/// </summary>
		/// <param name="mServerConfigHandle">M server handle specified in VistALink config.</param>
		/// <returns>Returns false if users cancels logon. True on successful logon.</returns>
		public bool Logon( string mServerConfigHandle )
		{
			if( mServerConfigHandle == null )
				throw( new ArgumentNullException( "mServerConfigHandle" ) );

			var msi = GlobalConfig.GetServerConnectionInfoByHandle( mServerConfigHandle );

			if( msi == null )
				throw( new VistALinkConfigurationException( SR.Exceptions.VistAServerNotFoundInAppConfigFile( mServerConfigHandle ) ) );

			return ConnectAndLogon( msi );
		}

		/// <summary>
		/// Top level logon wrapper performing thread safe connect &amp; logon and consuming class 
		/// event notification operation sequence.
		/// </summary>
		/// <param name="serverConnectionInfo">Server to connect to.</param>
		/// <returns>True if logon was successful.</returns>
		private bool ConnectAndLogon( ServerConnectionInfo serverConnectionInfo )
		{
			// No check for null argument is done here since this method is only callable via wrappers from the outside.

            _logger.Debug("ConnectAndLogon is called");
			lock( this )
			{
			    if (IsLoggedOn)
			    {
                    _logger.Warn("Rpc Broker Already LoggedOn");
			        throw (new InvalidOperationException(SR.Exceptions.RpcBrokerAlreadyLoggedOn()));
			    }


                if (!serverConnectionInfo.Equals(_cachedServerConnectionInfo))
                // if another server is used - cached access and verify codes are no good.
                {
                    _logger.Debug("another server is used - reset cached access and verify codes");
                    ResetCachedAccessCodes();
                }
			  
			    _cachedServerConnectionInfo = serverConnectionInfo; // refreshing cached server info because it can be expected by calling code.

				for(;;)
				{
					Connect( serverConnectionInfo );
					_isLoggedOn = LogonToVistAServer();

					if( !_isLoggedOn )
						return false;
					
					RetrieveUserDemographics();
					
					if( PerformPostLogonUserDemographicsCheck() )
					{
                        _logger.Debug("ConnectAndLogon is succeeded");
						OnConnectionStateChanged( new VistALinkConnectionStateChangedEventArgs( true ) );					
						return true;
					}

					_isLoggedOn = false; // Disabling event notification

					Close();

					ResetCachedAccessCodes();
				}
			}
		}

		/// <summary>
		/// Logs on to VistALink server. 
		/// </summary>
		/// <returns>Returns false if users cancels logon. True on successful logon.</returns>
		private bool LogonToVistAServer()
		{
			try
			{
				// Send setup and intro request
				var serverResponse = _vlConnection.SendReceiveMessage( new SecuritySetupAndIntroRequestMessage() );								

				// "kernel auto sign-on" case
				if( serverResponse is SecurityLogonSuccessResponseMessage )
					return true;

				// Verify that the message is of expected type
				if( !(serverResponse is SecuritySetupAndIntroResponseMessage) )
					throw( new UnexpectedServerResponseException( 
						SR.Exceptions.UnexpectedMessageReceivedDetailed( serverResponse.GetType().Name, typeof(SecuritySetupAndIntroResponseMessage).Name ), 
						serverResponse ) );

				_remoteServerInfo = ((SecuritySetupAndIntroResponseMessage)serverResponse).ServerInfo;

				// Requesting logon
				if( !HandleLogonAuthorization( out serverResponse ) )
					return CloseAndReturnFalse();

				// Normal successful logon for user with one division
				if( serverResponse is SecurityLogonSuccessResponseMessage )
					return true;

				// Verify code change required
				if( serverResponse is SecurityLogonVerifyCodeChangeRequiredResponseMessage && 
					!HandleVerifyCodeChange( out serverResponse, (SecurityLogonVerifyCodeChangeRequiredResponseMessage)serverResponse ) )
					return CloseAndReturnFalse();

				// Successful logon after verify code change
				if( serverResponse is SecurityUpdateVerifyCodeResponseMessage && 
					((SecurityUpdateVerifyCodeResponseMessage)serverResponse).ResponseStatus == ResponseStatus.Success ) // status check is not actually required
					return true;

				// Division required for user that have more than one division
				if( ( serverResponse is SecurityLogonDivisionRequiredResponseMessage ||
					serverResponse is SecurityUpdateVerifyCodeDivisionRequiredResponseMessage ) && 
					!HandleDivisionSelection( out serverResponse, serverResponse ) )
					return CloseAndReturnFalse();
				
				// Successful logon after selected division
				if( serverResponse is SecuritySelectDivisionResponseMessage && 
					((SecuritySelectDivisionResponseMessage)serverResponse).ResponseStatus == ResponseStatus.Success ) // status check is not actually required
					return true;

				throw( new UnexpectedServerResponseException( 
					SR.Exceptions.UnexpectedMessageReceivedDetailed( serverResponse.GetType().Name, typeof(SecuritySelectDivisionResponseMessage).Name ), 
					serverResponse ) );
			}
			catch( Exception )
			{
				Close();
				throw;
			}		
		}

		/// <summary>
		/// Handles initial authorization part of the logon process by exchanging VistALink messages with 
		/// VistA server and calling children-defined specific methods to query needed information from user,
		/// if needed. May reuse previously entered access and verify codes, if <see cref="ReuseAccessAndVerifyCodes"/> 
		/// flag is set.
		/// </summary>
		private bool HandleLogonAuthorization( out VistALinkMessage outServerResponse )
		{		
			outServerResponse = null;

		    for(;;)
			{
				VistASecurityCode accessCode, verifyCode;
				bool requestVerifyCodeChangeFlag;

				if( !GetAccessAndVerifyCodes( out accessCode, out verifyCode, out requestVerifyCodeChangeFlag ) )
					return false;

				var serverResponse = _vlConnection.SendReceiveMessage( new SecurityLogonRequestMessage( accessCode, verifyCode, requestVerifyCodeChangeFlag ) );
                if (serverResponse == null) throw new ArgumentNullException("serverResponse");

			    if( !(serverResponse is SecurityLogonFailureResponseMessage) )
				{
					outServerResponse = serverResponse;

					_cachedAccessCode = accessCode;
					_cachedVerifyCode = verifyCode;

					return true;
				}

				// Logon failure
				UINotifyAboutProblem( ((SecurityLogonFailureResponseMessage)serverResponse).Message );
			}
		}

		/// <summary>
		/// Gets current access and verify codes either from cached values or querying the user via 
		/// child-provided UI bound member.
		/// </summary>
		/// <param name="accessCode">Access code to return.</param>
		/// <param name="verifyCode">Verify code to return.</param>
		/// <param name="requestVerifyCodeChangeFlag">Returned flag indicating whether user wants to change verify code.</param>
		/// <returns>False if user elects to cancel sign-on. Otherwise - true.</returns>
		private bool GetAccessAndVerifyCodes( out VistASecurityCode accessCode, out VistASecurityCode verifyCode, out bool requestVerifyCodeChangeFlag )
		{
            _logger.Debug("Get current access and verify codes either from cached values or through UI");
			if( _reuseAccessAndVerifyCodes && _cachedAccessCode != null && _cachedVerifyCode != null )
			{
				accessCode = _cachedAccessCode;
				verifyCode = _cachedVerifyCode;
				requestVerifyCodeChangeFlag = false;
			}
			else
			{
				if( !UIGetAccessAndVerifyCodes( out accessCode, out verifyCode, out requestVerifyCodeChangeFlag ) )
					return false;

				if( accessCode == null )
					throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "accessCode" ) ) );

				if( verifyCode == null )
					throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "verifyCode" ) ) );
			}

			return true;
		}

		/// <summary>
		/// Handles change of user's VistA verify code part of logon process by exchanging VistALink messages with 
		/// VistA server and calling child-defined specific methods to query needed information from user.
		/// </summary>				
		/// <param name="outServerResponse">Latest response received from the server upon the end of verify code change procedure.</param>
		/// <param name="baseResponse">Initial server response opening verify code change procedure.</param>
		/// <returns>Last VistALink message received in process.</returns>
		private bool HandleVerifyCodeChange( out VistALinkMessage outServerResponse, SecurityLogonVerifyCodeChangeRequiredResponseMessage baseResponse )
		{
			if( baseResponse == null )
				throw( new ArgumentNullException( "baseResponse" ) );

			outServerResponse = null;

			for(;;)
			{				
				VistASecurityCode oldVerifyCode, newVerifyCode, newCheckVerifyCode;

				if( !GetVerifyCodeChangeParameters( out oldVerifyCode, out newVerifyCode, out newCheckVerifyCode ) )
					return false;

				var serverResponse = _vlConnection.SendReceiveMessage( new SecurityUpdateVerifyCodeRequestMessage( oldVerifyCode, newVerifyCode, newCheckVerifyCode ) );
			    if (serverResponse == null) throw new ArgumentNullException("serverResponse");

			    if( !(serverResponse is SecurityUpdateVerifyCodeResponseMessage && 
					((SecurityUpdateVerifyCodeResponseMessage)serverResponse).ResponseStatus == ResponseStatus.Failure ) )
				{
					_cachedVerifyCode = newVerifyCode;

					outServerResponse = serverResponse;					

					return true;
				}

				// Verify code change failure
				UINotifyAboutProblem( ((SecurityUpdateVerifyCodeResponseMessage)serverResponse).Message );
			}
		}

		/// <summary>
		/// Gets parameters required to change verify code by calling child-provided member querying user
		/// and validating returned values.
		/// </summary>
		/// <param name="oldVerifyCode">Old verify code to return.</param>
		/// <param name="newVerifyCode">New verify code to return.</param>
		/// <param name="newCheckVerifyCode">Confirmation for new verify code to return.</param>
		/// <returns>False if user elects to cancel sign-on. Otherwise - true.</returns>
		private bool GetVerifyCodeChangeParameters( out VistASecurityCode oldVerifyCode, out VistASecurityCode newVerifyCode, out VistASecurityCode newCheckVerifyCode )
		{
			if( !UIChangeVerifyCode( out oldVerifyCode, out newVerifyCode, out newCheckVerifyCode ) )
				return false;

			if( oldVerifyCode == null )
				throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "oldVerifyCode" ) ) );

			if( newVerifyCode == null )
				throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "newVerifyCode" ) ) );

			if( newCheckVerifyCode == null )
				throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "newCheckVerifyCode" ) ) );

			return true;
		}

		/// <summary>
		/// Handles division selection part of logon process by exchanging VistALink messages with 
		/// VistA server and calling children-defined specific methods to query needed information from user.
		/// </summary>
		/// <param name="serverResponse">Latest response received from the server upon the end of division selection procedure.</param>
		/// <param name="baseResponse">Initial server response opening division selection procedure.</param>		
		/// <returns>Last VistALink message received in process.</returns>
		private bool HandleDivisionSelection( out VistALinkMessage serverResponse, VistALinkMessage baseResponse )
		{			
			if( baseResponse == null )
				throw( new ArgumentNullException( "baseResponse" ) );

			serverResponse = null;

			var availableDivisions = ExtractListOfAvailableDivisions( baseResponse );
			
			for(;;)
			{
				DivisionInfo userSelectedDivision;

				if( !GetDivisionSelection( availableDivisions, out userSelectedDivision ) )
					return false;

				var _serverResponse = _vlConnection.SendReceiveMessage( new SecuritySelectDivisionRequestMessage( userSelectedDivision.Ien ) );

				if( !(_serverResponse is SecuritySelectDivisionResponseMessage &&
					((SecuritySelectDivisionResponseMessage)_serverResponse).ResponseStatus == ResponseStatus.Failure ) )
				{
					serverResponse = _serverResponse;
					return true;
				}

				// Select division failure
				UINotifyAboutProblem( ((SecuritySelectDivisionResponseMessage)_serverResponse).Message );
			}
		}

		/// <summary>
		/// Used for sign-on, extracts list of VistA divisions available to user from supplied VistALink message.
		/// </summary>
		/// <param name="sourceMessage">Source <see cref="VistALinkMessage"/> to extract list of available divisions from.</param>
		/// <returns>List of divisions available for user in VistA.</returns>
		private DivisionInfoCollection ExtractListOfAvailableDivisions( VistALinkMessage sourceMessage )
		{
			if( sourceMessage is SecurityLogonDivisionRequiredResponseMessage )
				return ((SecurityLogonDivisionRequiredResponseMessage)sourceMessage).Divisions;
			
			if(	sourceMessage is SecurityUpdateVerifyCodeDivisionRequiredResponseMessage )
				return ((SecurityUpdateVerifyCodeDivisionRequiredResponseMessage)sourceMessage).Divisions;
			
			throw( new VistALinkException( SR.Exceptions.UnableToExtractDivisionListFromVLMessage( sourceMessage.GetType().Name ) ) );
		}

		/// <summary>
		/// Gets division selection during a sign-on procedure by querying user via child-provided member
		/// and validating its output.
		/// </summary>
		/// <param name="availableDivisions">List of divisions available for VistA user.</param>
		/// <param name="userSelectedDivision">Division selected by user to return.</param>
		/// <returns>False if user elects to cancel sign-on. Otherwise - true.</returns>
		private bool GetDivisionSelection( DivisionInfoCollection availableDivisions, out DivisionInfo userSelectedDivision )
		{
			if( availableDivisions == null )
				throw( new ArgumentNullException( "availableDivisions" ) );

            _logger.Debug("GetDivisionSelection is called.");
			if( !UISelectDivision( availableDivisions, out userSelectedDivision ) )
				return false;

			if( userSelectedDivision == null )
				throw( new VistALinkException( SR.Exceptions.CallerDefinedUIMethodReturnedNullInParameter( GetType().Name, "userSelectedDivision" ) ) );

			return true;
		}

		/// <summary>
		/// Executes RPC request and returns the result. 
		/// </summary>
		/// <param name="request">RPC request to execute.</param>
		/// <returns>Remote procedure return value extracted from <see cref="RpcResponseMessage"/>.</returns>
		public string ExecuteRpc( RpcRequest request )
		{
            // CR2761
            // Wrap all exceptions coming from this function to recoverable exceptions.
            try
            {
                if( request == null ) 
                    throw( new ArgumentNullException( "request" ) );

                return ExecuteRpcReturnResponseObject( request ).RawRpcResult;
            }
            catch(VistALinkNetworkErrorException) // this exception is already marked as recoverable
            {
                _logger.Error("ExecuteRpc failed with VistALinkNetworkErrorException (recoverable). Break icon and re-throw.");
                //Break connection icon
                OnConnectionStateChanged(new VistALinkConnectionStateChangedEventArgs(false)); //CR 3304
                throw;
            }
            catch(Exception ex)
            {
                _logger.Error("ExecuteRpc failed with generic exception. Wrap it in recoverable and re-throw.", ex);
                // re-throw new exception type. Necessary to make all occurred problems recoverable.
                throw new VistALinkClientRPCException("VistALink client RPC failed", ex);
            }
		}

		/// <summary>
		/// Checks that <see cref="RpcBroker"/> is logged on to VistA, if not - throws <see cref="InvalidOperationException"/>
		/// with an appropriate message. 
		/// </summary>
		private void RequireLoggedOnState()
		{
			if( !IsLoggedOn )
				throw( new InvalidOperationException( SR.Exceptions.RpcBrokerNotLoggedOnToVistA() ) );
		}

		/// <summary>
		/// Resets cached access and verify codes used for silent re-logon.
		/// </summary>
		private void ResetCachedAccessCodes()
		{
			_cachedAccessCode = _cachedVerifyCode = null;
		}

		/// <summary>
		/// Executes RPC request and returns an <see cref="RpcResponseMessage"/> object.
		/// </summary>
		/// <param name="request">RPC request to execute.</param>
		/// <returns>RPC response.</returns>
		private RpcResponseMessage ExecuteRpcReturnResponseObject( RpcRequest request )
		{
			if( request == null )
				throw( new ArgumentNullException( "request" ) );

			RequireLoggedOnState();

			return (RpcResponseMessage)_vlConnection.SendReceiveMessage( new RpcRequestMessage( request ) );
		}

		/// <summary>
		/// Retrieves user system, name and logon division info from VistA system. 
		/// </summary>
		private void RetrieveUserDemographics()
		{
			RequireLoggedOnState();

			var response = (SecurityGetUserDemographicsResponseMessage)_vlConnection.SendReceiveMessage( 
				new SecurityGetUserDemographicsRequestMessage() );

			_remoteUserInfo = response.UserInfo;
			_remoteUserNameInfo = response.UserNameInfo;
			_remoteUserLogonDivisionInfo = response.UserDivisionInfo;
		}

		/// <summary>
		///		Checks VistALink availability by using underlying connection's diagnostic method sending
		///		heartbeat message to the remote server. 
		/// </summary>
		/// <returns>True if connection is available. Otherwise - false.</returns>	
		/// Note: it doesn't really checks for vistalink server availability only. It also checks if user logged in.
		/// which makes it similar to IsLoggedOn event.
		public bool CheckHeartbeatGetStatus()
		{
            // CR2761 and CR3304
            // Wrap all exceptions coming from this function to recoverable exceptions.
            // If last heart beat failed IsConnected will return false.
            return IsLoggedOn;
		}

		/// <summary>
		/// The property indicates if the <see cref="RpcBroker"/> is logged on to the 
		/// remote server. 
		/// </summary>
		public bool IsLoggedOn
		{
			get
			{
				return IsConnected && _isLoggedOn;
			}
		}

		/// <summary>
		/// The property indicates if the <see cref="RpcBroker"/> is connected to the 
		/// remote server. 
		/// </summary>
		private bool IsConnected
		{
			get
			{
				return _vlConnection != null && _vlConnection.IsConnected;
			}
		}

		/// <summary>
		/// Connection information for active (connected to) remote M server.
		/// </summary>
		public ServerConnectionInfo ServerConnectionInfo
		{
			get
			{
				return IsConnected ? _vlConnection.ServerConnectionInfo : null;
			}
		}

		/// <summary>
		/// Remote (logged on to) server information. 
		/// </summary>
		public RemoteServerSetupAndIntroInfo RemoteServerSetupAndIntroInfo
		{
			get
			{				
				return IsConnected ? _remoteServerInfo : null;
			}
		}

		/// <summary>
		/// Logged on user VistA system information.
		/// </summary>
		public RemoteUserInfo RemoteUserInfo
		{
			get
			{
				RequireLoggedOnState();

				return _remoteUserInfo;
			}
		}

		/// <summary>
		/// Logged on user VistA name information.
		/// </summary>
		public RemoteUserNameInfo RemoteUserNameInfo
		{
			get
			{
				RequireLoggedOnState();

				return _remoteUserNameInfo;
			}
		}

		/// <summary>
		/// Division information for logged on user. 
		/// </summary>
		public DivisionInfo RemoteUserLogonDivisionInfo
		{
			get
			{
				RequireLoggedOnState();

				return _remoteUserLogonDivisionInfo;
			}
		}

		/// <summary>
		/// Specifies whether supplied access and verify codes 
		/// should be reused for subsequent logons after first logon and logoff.
		/// Can be set at any time but will be effective for the next logon only 
		/// if set before previous logon. Does not guarantee that client will not 
		/// be required to re-enter access and verify codes, since they may be changed 
		/// on server side or not available due to setting this flag after logon.
		/// </summary>
		public bool ReuseAccessAndVerifyCodes
		{
			get
			{
				return _reuseAccessAndVerifyCodes;
			}
			set
			{
				_reuseAccessAndVerifyCodes = value;
			}
		}

		/// <summary>
		/// Logs out from M server and disconnects. 
		/// </summary>
		public void Close()
		{
			Dispose();
		}	

		/// <summary>
		/// Shortcut wrapper closing the connection and returning false.
		/// </summary>
		/// <returns>Always returns false.</returns>
		private bool CloseAndReturnFalse()
		{
			Close();

			return false;
		}

		/// <summary>
		/// Standard finalization method.
		/// </summary>
		~RpcBroker()
		{
			Dispose( false );
		}

		/// <summary>
		/// Standard method needed to implement <see cref="IDisposable"/> interface. 
		/// Does some clean-up needed to terminate underlying connection nicely. 
		/// </summary>
		public void Dispose()
		{
			//Adding code to submit Logout request to VistA -- CR3121
			//Without this code multiple "session" will remain open for a particular user/ip, which can cause problems if 
			//the connections are limited on the VistA end via the Kernel Parameters File
            _logger.Debug("Dispose is called");
			if (_vlConnection != null)
			{
				if (_vlConnection.IsConnected)
				{
					_vlConnection.SendReceiveMessage( new SecurityLogoutRequestMessage() );
				}
			}

			GC.SuppressFinalize( this );
			Dispose( true );
		}


		/// <summary>
		/// Standard implementation of the dispose method. 
		/// </summary>
		/// <param name="disposing">
		///		Flag indicating if the disposition was invoked explicitly under normal 
		///		conditions (true) or forced upon object disposal (false).
		///	</param>
		private void Dispose( bool disposing )
		{
			lock( this )
			{
				try
				{
					if( !disposing )
						return;

					if( _isLoggedOn )
					{
						_isLoggedOn = false;
					}

				    if (_vlConnection == null) return;

				    _vlConnection.Close();
				    _vlConnection.ConnectionStateChanged -= ConnectionStateChangedHandler;
				}
				finally
				{
					ResetConnectionStateVars();
				}
			}
		}

		/// <summary>
		/// Resets member variables defining connection state. 
		/// </summary>
		private void ResetConnectionStateVars()
		{
			_isLoggedOn = false;
			_vlConnection = null;

			_remoteServerInfo = null;
			_remoteUserInfo = null;
			_remoteUserNameInfo = null;
			_remoteUserLogonDivisionInfo = null;
		}

		/// <summary>
		/// Handles underlying <see cref="VistALinkClientConnection"/> connection state change 
		/// event and notifies VistALink-consuming classes of connection state change.
		/// </summary>
		/// <param name="sender">Event sender object</param>
		/// <param name="e">Connection state change event arguments.</param>
		private void ConnectionStateChangedHandler( object sender, VistALinkConnectionStateChangedEventArgs e )
		{
            //VistALink-consuming classes should be notified only when user logged on.
            //This event is raised RpcBroker itself so suppress connection available event here.
		    if (!e.IsAvailable)
		    {
                OnConnectionStateChanged(e);   
		    }
		}

		/// <summary>
		/// Notifies VistALink-consuming classes of VistALink availability change.
        /// Note: it doesn't really notify about VistALink availability. It should notify when user logged on.
		/// </summary>
		/// <param name="e">Connection state change event arguments.</param>
		protected virtual void OnConnectionStateChanged( VistALinkConnectionStateChangedEventArgs e )
		{
            _logger.Debug(string.Format("Connection state changed: {0}", e.IsAvailable));
		    if (ConnectionStateChanged != null)
		    {
		        ConnectionStateChanged(this, e);
		    }
		}

		/// <summary>
		/// Subclasses must implement this method to use specific UI and get 
		/// access and verify codes from user during the logon process. 
		/// Method is expected to return false if user cancels logon. 
		/// </summary>
		/// <param name="accessCode">VistA access code entered by user.</param>
		/// <param name="verifyCode">VistA verify code entered by user.</param>
		/// <param name="requestVerifyCodeChangeFlag">Flag indicating whether user wants to change verify code.</param>
		/// <returns>True if user have chosen to change verify code and continue VistA sign-on. False if user cancels logon.</returns>
		protected abstract bool UIGetAccessAndVerifyCodes( out VistASecurityCode accessCode, out VistASecurityCode verifyCode, out bool requestVerifyCodeChangeFlag );

		/// <summary>
		/// Subclasses must implement this method to use specific UI and get  
		/// division selection from user during the logon process. 
		/// Method is expected to return false if user cancels logon. 
		/// </summary>
		/// <param name="divisionList">List of divisions available for user to select from.</param>
		/// <param name="userSelectedDivision">Division user has selected.</param>
		/// <returns>True if user have chosen to change verify code and continue VistA sign-on. False if user cancels logon.</returns>
		protected abstract bool UISelectDivision( DivisionInfoCollection divisionList, out DivisionInfo userSelectedDivision );
			
		/// <summary>
		/// Subclasses must implement this method to use specific UI and 
		/// change user verify code during the logon process 
		/// (method is expected to get old and new verify codes from user)
		/// Method is expected to return false if user cancels logon. 
		/// </summary>
		/// <param name="oldVerifyCode">Old VistA verify code.</param>
		/// <param name="newVerifyCode">New VistA verify code.</param>
		/// <param name="newCheckVerifyCode">New VistA verify code (entered twice to ensure codes match).</param>
		/// <returns>True if user have chosen to change verify code and continue VistA sign-on. False if user cancels logon.</returns>
		protected abstract bool UIChangeVerifyCode( out VistASecurityCode oldVerifyCode, out VistASecurityCode newVerifyCode, out VistASecurityCode newCheckVerifyCode );

		/// <summary>
		/// Subclasses must implement this method to use specific UI and 
		/// notify user about problems that may occur during VistA sign-on.
		/// </summary>
		/// <param name="message">Error message to notify user about.</param>
		protected abstract void UINotifyAboutProblem( string message );

		/// <summary>
		/// This method may be used by subclasses to perform additional validation/operations for user identity
		/// after the logon to VistA and retrieval of user demographics, but before event notification. If it 
		/// returns false, logon operation will be cancelled. 
		/// </summary>
		/// <returns>True if user demographics is verified. Otherwise - false.</returns>
		protected virtual bool PerformPostLogonUserDemographicsCheck()
		{
			return true;
		}
	}
}
